import os
from configparser import ConfigParser

from event.sync import EventDispatcher


class ConfigFile(EventDispatcher, ConfigParser):
    POSITION_TOP_LEFT = 'tl'
    POSITION_TOP_RIGHT = 'tr'
    POSITION_BOTTOM_LEFT = 'bl'
    POSITION_BOTTOM_RIGHT = 'br'

    ICON_TRAY_DEFAULT = ':/icon.png'
    ICON_DROPPER_EMPTY_DEFAULT = ':/dropper_empty.png'
    ICON_DROPPER_FULL_DEFAULT = ':/dropper_full.png'

    def __init__(self, path):
        EventDispatcher.__init__(self, 'changed')
        ConfigParser.__init__(self)
        self.path = path
        if os.path.exists(path):
            self.load()

    def get_position(self):
        return self.get('ui', 'position', fallback=self.POSITION_BOTTOM_RIGHT)

    def set_position(self, position):
        assert position in (
            self.POSITION_TOP_LEFT, self.POSITION_TOP_RIGHT, self.POSITION_BOTTOM_LEFT, self.POSITION_BOTTOM_RIGHT
        )

        if not self.has_section('ui'):
            self.add_section('ui')
        self.set('ui', 'position', position)

    def get_offset(self):
        return (
            self.getint('ui', 'offset_x', fallback=16),
            self.getint('ui', 'offset_y', fallback=16)
        )

    def set_offset(self, offset_x, offset_y):
        if not self.has_section('ui'):
            self.add_section('ui')

        self.set('ui', 'offset_x', str(offset_x))
        self.set('ui', 'offset_y', str(offset_y))

    def get_icon_size(self):
        return (
            self.getint('ui', 'icon_width', fallback=32),
            self.getint('ui', 'icon_height', fallback=32)
        )

    def set_icon_size(self, icon_width, icon_height):
        if not self.has_section('ui'):
            self.add_section('ui')

        self.set('ui', 'icon_width', str(icon_width))
        self.set('ui', 'icon_height', str(icon_height))

    def get_tray_icon(self):
        return self.get('ui', 'tray_icon', fallback=self.ICON_TRAY_DEFAULT)

    def set_tray_icon(self, icon):
        if not self.has_section('ui'):
            self.add_section('ui')

        self.set('ui', 'tray_icon', icon)

    def get_empty_icon(self):
        return self.get('ui', 'empty_icon', fallback=self.ICON_DROPPER_EMPTY_DEFAULT)

    def set_empty_icon(self, icon):
        if not self.has_section('ui'):
            self.add_section('ui')

        self.set('ui', 'empty_icon', icon)

    def get_full_icon(self):
        return self.get('ui', 'full_icon', fallback=self.ICON_DROPPER_FULL_DEFAULT)

    def set_full_icon(self, icon):
        if not self.has_section('ui'):
            self.add_section('ui')

        self.set('ui', 'full_icon', icon)

    def load(self):
        self.read(self.path)

    def save(self):
        with open(self.path, 'w') as f:
            self.write(f)

        self.dispatch('changed')
