from event.sync import EventDispatcher
from .proxy import OP_BS_EMU, OP_DEV_ID, OP_DPM_EMU, OP_TR_EMU


class CDEmuDevice(EventDispatcher):
    def __init__(self, device_id, cdemu):
        super().__init__(
            'change'
        )
        self.cdemu = cdemu
        self._device_id = device_id
        self._mapped_sr = None
        self._mapped_sg = None
        self._img = False
        self._dpm = False
        self._tr = False
        self._bs = False
        self._vendor = None
        self._product = None
        self._revision = None
        self._vendor_specific = None

    def get_name(self):
        return 'Device {}'.format(self._device_id)

    @property
    def mapped_sr(self):
        return self._mapped_sr

    @property
    def mapped_sg(self):
        return self._mapped_sg

    def get_img(self):
        return self._img

    def load(self, img):
        self.cdemu.load(self, img)

    def unload(self):
        self.cdemu.unload(self)

    def get_dpm(self):
        """
        get current dpm emulation state
        :return: True if dpm emulation is on, False otherwise
        """
        return self._dpm

    def set_dpm(self, dpm):
        """
        set dpm emulation state
        :param dpm: True to enable emulation, False to disable
        """
        self.cdemu.set_device_option(self._device_id, OP_DPM_EMU, dpm)

    def get_tr(self):
        """
        get current transfer rate emulation state
        :return: True if transfer rate emulation is enabled, False otherwise
        """
        return self._tr

    def set_tr(self, tr):
        """
        set transfer rate emulation state
        :param tr: True to enable emulation, False to disable
        """
        self.cdemu.set_device_option(self._device_id, OP_TR_EMU, tr)

    def get_bs(self):
        """
        get current bad sector emulation state
        :return: True if bad sector emulation is enabled, False otherwise
        """
        return self._bs

    def set_bs(self, bs):
        """
        set bad sector emulation state
        :param bs: True to enable emulation, False to disable
        """
        self.cdemu.set_device_option(self._device_id, OP_BS_EMU, bs)

    def get_id(self):
        """
        get current device id
        :return: tuple(str(vendor_id), str(product_id), str(revision), str(vendor_specific))
        """
        return (self._vendor, self._product, self._revision, self._vendor_specific)

    def set_id(self, vendor, product, revision, vendor_specific):
        """
        set device id
        :param vendor: vendor_id - 8 character string
        :param product: product_id - 16 character string
        :param revision: - 4 character string
        :param vendor_specific: - 20 character string
        """
        self.cdemu.set_device_option(self._device_id, OP_DEV_ID, (vendor, product, revision, vendor_specific))
