import os

from PyQt5 import QtCore
from PyQt5.QtCore import QSize
from PyQt5.QtWidgets import QWidget, QFileDialog

from .ui.main_settings import Ui_Form


class MainSettingsWidget(Ui_Form, QWidget):
    def __init__(self, config):
        """

        :type config: config.config.ConfigFile
        """
        super().__init__()
        self.config = config
        self.last_directory = None
        self.setupUi(self)

        self.trayDefaultB.clicked.connect(self.on_tray_default_click)
        self.emptyDefaultB.clicked.connect(self.on_empty_default_click)
        self.fullDefaultB.clicked.connect(self.on_full_default_click)

        self.trayCustomB.clicked.connect(self.on_tray_custom_click)
        self.emptyCustomB.clicked.connect(self.on_empty_custom_click)
        self.fullCustomB.clicked.connect(self.on_full_custom_click)

        position = self.config.get_position()
        if position == self.config.POSITION_TOP_LEFT:
            self.topLeftRB.setChecked(True)
        if position == self.config.POSITION_TOP_RIGHT:
            self.topRightRB.setChecked(True)
        if position == self.config.POSITION_BOTTOM_LEFT:
            self.bottomLeftRB.setChecked(True)
        if position == self.config.POSITION_BOTTOM_RIGHT:
            self.bottomRightRB.setChecked(True)

        offset = self.config.get_offset()
        icon_size = self.config.get_icon_size()

        self.hOffsetSB.setValue(offset[0])
        self.vOffsetSB.setValue(offset[1])

        self.iwidthSB.setValue(icon_size[0])
        self.iheightSB.setValue(icon_size[1])

        self.iwidthSB.editingFinished.connect(self.resize_icons)
        self.iheightSB.editingFinished.connect(self.resize_icons)
        self.resize_icons()

        self.tray_icon = self.config.get_tray_icon()
        self.empty_icon = self.config.get_empty_icon()
        self.full_icon = self.config.get_full_icon()

        self.update_icons()

    def update_icons(self):
        self.trayLabel.setText("<html><head/><body><p><img src=\"" + self.tray_icon + "\"/></p></body></html>")
        self.emptyLabel.setText("<html><head/><body><p><img src=\"" + self.empty_icon + "\"/></p></body></html>")
        self.fullLabel.setText("<html><head/><body><p><img src=\"" + self.full_icon + "\"/></p></body></html>")

    def on_tray_default_click(self):
        self.tray_icon = self.config.ICON_TRAY_DEFAULT
        self.update_icons()

    def on_empty_default_click(self):
        self.empty_icon = self.config.ICON_DROPPER_EMPTY_DEFAULT
        self.update_icons()

    def on_full_default_click(self):
        self.full_icon = self.config.ICON_DROPPER_FULL_DEFAULT
        self.update_icons()

    def on_tray_custom_click(self):
        file = self.show_open_dialog()
        if file:
            self.tray_icon = file
            self.update_icons()

    def on_empty_custom_click(self):
        file = self.show_open_dialog()
        if file:
            self.empty_icon = file
            self.update_icons()

    def on_full_custom_click(self):
        file = self.show_open_dialog()
        if file:
            self.full_icon = file
            self.update_icons()

    def show_open_dialog(self):
        open_dialog = QFileDialog()

        if self.last_directory:
            open_dialog.setDirectory(self.last_directory)

        open_dialog.setAcceptMode(open_dialog.AcceptOpen)
        if open_dialog.exec_() == QFileDialog.Accepted:
            file = open_dialog.selectedFiles()[0]
            self.last_directory = os.path.abspath(file)
            return file
        return None

    def resize_icons(self):
        icon_width = self.iwidthSB.value()
        icon_height = self.iheightSB.value()
        self.trayLabel.setMinimumSize(icon_width, icon_height)
        self.trayLabel.setBaseSize(icon_width, icon_height)
        self.trayLabel.setMaximumSize(icon_width, icon_height)

        self.emptyLabel.setMinimumSize(icon_width, icon_height)
        self.emptyLabel.setBaseSize(icon_width, icon_height)
        self.emptyLabel.setMaximumSize(icon_width, icon_height)

        self.fullLabel.setMinimumSize(icon_width, icon_height)
        self.fullLabel.setBaseSize(icon_width, icon_height)
        self.fullLabel.setMaximumSize(icon_width, icon_height)

        self.trayDefaultB.setIconSize(QSize(icon_width, icon_height))
        self.emptyDefaultB.setIconSize(QSize(icon_width, icon_height))
        self.fullDefaultB.setIconSize(QSize(icon_width, icon_height))


    def update_config(self):
        position = None
        if self.topLeftRB.isChecked():
            position = self.config.POSITION_TOP_LEFT
        if self.topRightRB.isChecked():
            position = self.config.POSITION_TOP_RIGHT
        if self.bottomLeftRB.isChecked():
            position = self.config.POSITION_BOTTOM_LEFT
        if self.bottomRightRB.isChecked():
            position = self.config.POSITION_BOTTOM_RIGHT

        self.config.set_position(position)
        self.config.set_icon_size(self.iwidthSB.value(), self.iheightSB.value())
        self.config.set_offset(self.hOffsetSB.value(), self.vOffsetSB.value())
        self.config.set_tray_icon(self.tray_icon)
        self.config.set_empty_icon(self.empty_icon)
        self.config.set_full_icon(self.full_icon)
